#!/bin/bash
set -e

############################################
# PART 1 — BASIC SERVER + NODE + X-UI SETUP
############################################

echo "==> Setting root password..."
echo "root:Moqol2424!!" | chpasswd

echo "==> Updating package index..."
apt update

echo "==> Installing base packages..."
apt install -y curl git vnstat

echo "==> Installing Node.js LTS..."
curl -fsSL https://deb.nodesource.com/setup_lts.x | bash -
apt-get install -y nodejs

echo "==> Installing pm2 globally..."
npm install -g pm2

echo "==> Verifying Node / npm / pm2 versions..."
node_version=$(node -v)
npm_version=$(npm -v)
pm2_version=$(pm2 -v)

echo "Node.js version: $node_version"
echo "npm version:     $npm_version"
echo "pm2 version:    $pm2_version"

echo "==> Cloning x-ui project..."
cd /usr/local/

if [ ! -d "x-ui" ]; then
  git clone http://git.fcfglobal.co/root/x-ui.git
else
  echo "x-ui already exists, skipping clone."
fi

cd x-ui

echo "==> Setting permissions..."
chmod 777 xray

echo "==> Installing project dependencies..."
npm install

echo "==> Starting project with pm2..."
pm2 startup
pm2 start app.json
pm2 save

echo "==> x-ui setup completed successfully."

############################################
# PART 2 — MTProxy with Fake-TLS (translate.goo)
############################################

echo
echo "============================================"
echo " Starting MTProxy installation and setup..."
echo "============================================"

### MTProxy configuration
SECRET_HEX="e9a4f23b1d768c04a8d7f39120ca5b6e"
SPONSOR_TAG="f2f3927f0e461d74d7f06bbddfacd8a7"
FAKETLS_DOMAIN="media.steampowered.com"

PROXY_PORT="9443"
STATS_PORT="8888"
MTProxy_USER="mtproxy"
MTProxy_DIR="/etc/mtproxy"
MTProxy_BIN="/usr/local/bin/mtproto-proxy"
SERVICE_FILE="/etc/systemd/system/mtproxy.service"

echo "==> Installing MTProxy dependencies..."
apt install -y build-essential libssl-dev zlib1g-dev xxd

echo "==> Building MTProxy (if not already installed)..."
if [ ! -x "$MTProxy_BIN" ]; then
  cd /opt
  if [ ! -d "MTProxy" ]; then
    git clone https://github.com/TelegramMessenger/MTProxy
  fi
  cd MTProxy
  make
  cp objs/bin/mtproto-proxy "$MTProxy_BIN"
else
  echo "MTProxy binary already exists."
fi

echo "==> Creating MTProxy system user (if needed)..."
if ! id "$MTProxy_USER" &>/dev/null; then
  useradd -r -M -s /usr/sbin/nologin "$MTProxy_USER"
fi

echo "==> Preparing MTProxy configuration directory..."
mkdir -p "$MTProxy_DIR"
cd "$MTProxy_DIR"

echo "==> Downloading Telegram proxy secrets..."
curl -s https://core.telegram.org/getProxySecret -o proxy-secret
curl -s https://core.telegram.org/getProxyConfig -o proxy-multi.conf

chown -R "$MTProxy_USER":"$MTProxy_USER" "$MTProxy_DIR"

echo "==> Creating or updating systemd service for MTProxy..."

cat > "$SERVICE_FILE" <<EOF
[Unit]
Description=Telegram MTProxy with Fake-TLS
After=network.target

[Service]
Type=simple
WorkingDirectory=$MTProxy_DIR
ExecStart=$MTProxy_BIN \\
  -u $MTProxy_USER \\
  -p $STATS_PORT \\
  -H $PROXY_PORT \\
  -D $FAKETLS_DOMAIN \\
  -S $SECRET_HEX \\
  -P $SPONSOR_TAG \\
  --aes-pwd proxy-secret proxy-multi.conf \\
  -M 3
Restart=on-failure
LimitNOFILE=100000

[Install]
WantedBy=multi-user.target
EOF

echo "==> Reloading systemd and restarting MTProxy..."
systemctl daemon-reload
systemctl enable mtproxy.service
systemctl restart mtproxy.service

echo "==> MTProxy service status:"
systemctl status mtproxy.service --no-pager || true

############################################
# FINAL INFO
############################################

FAKETLS_HEX=$(echo -n "$FAKETLS_DOMAIN" | xxd -ps)
SERVER_IP=$(hostname -I | awk '{print $1}')

echo
echo "============================================"
echo " ALL SERVICES INSTALLED SUCCESSFULLY"
echo "============================================"
echo "MTProxy Port:        $PROXY_PORT"
echo "Hex Secret:         $SECRET_HEX"
echo "Sponsor Tag:        $SPONSOR_TAG"
echo "Fake-TLS Domain:    $FAKETLS_DOMAIN"
echo
echo "Telegram MTProxy link example:"
echo "https://t.me/proxy?server=$SERVER_IP&port=$PROXY_PORT&secret=ee${SECRET_HEX}${FAKETLS_HEX}"
echo
echo "You may replace the IP with your own domain name."
echo
